; Mapped.asm - Memory mapped files
;
; This console application demonstrates how to use memory-mapped files (also commonly referred
; to as mapped files or file mapping), and is also a useful utility which we'll use with
; programs that have resources (menus, dialog boxes, etc)
;
; It reads in an input file (specified on the command line), and generates an assembler header
; file with equates for all the resource IDs, etc.
;
; This utility is used for the examples with resource files, and a copy of the output
; executable can be found in the Bin directory, renamed h2ash.
;
; File mapping is an extremely useful ability of Win32. Essentially, what it amounts to is
; being able to treat a file as if it was residing in memory rather than on disk. As you can
; no doubt imagine, this can be very powerful - we can use normal memory access code to
; access any part of the file and don't have to worry about _how_ the file is actually
; mapped into memory - Windows takes care of it all for us.
;
; Surely there has to be some drawback to this great function? Well there is, you can't
; easily map a file into memory and then expand or contact its size, though you can
; write to it if the size stays unchanged.
;
; The process operates like:
;
; 1. Open a file as per usual
; 2. Map the file into memory
; 3. Create a view of the mapped file
; 4. Close the view(s)
; 5. Unmap the file
; 6. Close the file
;
; All of this will be explained in the example below
;
; NB: Because of the large local buffers, don't forget that the linker needs to
; be told to reserve some stack space for us.
;

%define _WINCONSOLE_
%define _WINNT_
%define _WINIO_
%include "Gaz\Win32\Include\Windows.inc"

[BITS 32]
[section .text]

procglobal main
	ddlocal		_hStdIn, _hStdOut
	ddlocal		_lpFile1
	ddlocal		_hFile1, _hFile1M, _hFile1V, _hFile2
	ddstatic	_ddFile1Size, _ddLineSize
	ddstatic	_argc, _tmp
	bufflocal	_argv, 256*4, _szargv, 4096*4, _szFile2, 512, _szLine, 512
	endlocals
	;
	sc GetStdHandle, STD_INPUT_HANDLE
	mov	._hStdIn, eax
	sc GetStdHandle, STD_OUTPUT_HANDLE
	mov	._hStdOut, eax
	;
	; Parse the command line using the routiner from Cmdline.asm
	;
	sc GetCommandLine			; get pointer to the command line
	mov	._argc, dword -1		; _argc holds the argument count, starting at -1
						; because the first paramter is the program itself
	mov	esi, ._argv			; esi points to the argument pointer list
	mov	edi, ._szargv			; edi points to the argument buffer itself
	mov	[esi], edi			; first argument points to the start of the buffer
	add	esi, 4				; onto the next argument pointer
	while [eax], ne, byte 0			; loop until the end of the string is found
		if [eax], e, byte ' '		; argument delimiter? (space)
			while [eax], e, byte ' '
				inc	eax	; ignore any multiple spaces
			wend
			cmp	[eax], byte 0	; end of command line? (if no arguments this is true)
			jz	_cmdline_end	; yes, end the search here
			mov	[edi], byte 0	; no, add a null terminator to the string
			inc	edi		; onto the next byte in the buffer
			mov	ebx, edi	; get the address so we can align to a dword boundary
			and	ebx, 3		; mask off all but bottom two bits
			sub	ebx, 4		; subtract 4, so 4 = 0, 3 = -1, 2 = -2, 1 = -3, 0 = -4
			and	ebx, 3		; and mask off all but bottom two bits to get 0, 1, 2, 3 or 4
			add	edi, ebx	; and add to the buffer address to dword align it
			mov	[esi], edi	; set the pointer for the next argument
			add	esi, 4		; onto the next argument pointer
			inc	._argc		; increase the argument count
		else
			mov	bl, [eax]	; no, copy byte to buffer
			mov	[edi], bl
			inc	eax		; onto the next character in the command line
			inc	edi		; and in the buffer
		endif
	wend
_cmdline_end:
	inc	._argc				; add in the final argument
	mov	[esi], dword 0			; add an end of argument list marker
	;
	; Work out the input file, we assume the command line contains only one parameter - the
	; input file - so check we've only got one parameter
	;
	xor	eax, eax
	if ._argc, ne, 0
		mov	eax, 1
	endif
	if ._argc, g, 1
		mov	eax, 0
	endif
	if eax, e, 1
		;
		; generate the output filename, which is:
		; <input filename without extension>.ash
		;
		mov	esi, ._argv		; esi -> parameter list
		mov	eax, [esi + 4]		; +4 because first parameter is always the program itself
		mov	._lpFile1, eax		; save address of the filename
		while byte [eax], ne, 0
			inc	eax		; move to the end of the parameter
		wend
		dec	eax			; eax -> last character in parameter
		while byte [eax], ne, 0
			if byte [eax], e, '.'	; is this character the separator?
				jmp	WinMain_fname	; yes, stop looking
			endif
			dec	eax		; no, onto previous character in parameter
		wend
		;
		; no separator found, so display an error message and end
		;
		TEXTlocal _szErrorNoSeparator, 'No file extension found!'
		TEXTlocal _szErrorNoSeparatorTemp, 0
		mov	eax, .._szErrorNoSeparator
		sub	eax, ._szErrorNoSeparator
		sc WriteConsole, ._hStdOut, ._szErrorNoSeparator, eax, ._szErrorNoSeparatorTemp, NULL
		jmp	WinMain_end
WinMain_fname:
		;
		; now copy the <input filename without extension> to the buffer
		; eax currently -> separator character
		;
		mov	esi, ._argv		; esi -> parameter list
		mov	esi, [esi + 4]		; +4 because first parameter is always the program itself
		mov	edi, ._szFile2		; edi -> buffer
		while byte [esi], ne, '.'
			mov	bl, [esi]	; copy bytes from the source to the destination
			inc	esi		; until we reach the separator
			mov	[edi], bl
			inc	edi
		wend
		mov	[edi], dword '.ash'	; add in the file extension and separator
		mov	[edi + 4], dword 0	; and the null terminator
		;
		; Open file1 in read-only mode, with no sharing (ie exclusive)
		;
		sc CreateFile, ._lpFile1, GENERIC_READ, NULL, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL
		mov	._hFile1, eax
		;
		; Get the size of the file
		;
		sc GetFileSize, eax, NULL
		mov	._ddFile1Size, eax
		;
		; Open file2 for writing
		;
		sc CreateFile, ._szFile2, GENERIC_WRITE, NULL, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL
		mov	._hFile2, eax
		;
		; Create a file mapping for file 1 with CreateFileMapping, taking six parameters:
		;
		;   1. File handle of file to map
		;   2. Address of security attributes (not required)
		;   3. Protection for the file views (mapped later)
		;   4. High 32 bits of maximum size to map (zero = all)
		;   5. Low 32 bits
		;   6. Pointer to a name for the map (not required)
		;
		; All this call does is tell Windows that we have opened a file and we would like
		; to be able to access it as if it was mapped into memory. Note that this doesn't
		; actually enable us to access the file yet, just tells Windows that we want it
		; mapped into memory
		;
		sc CreateFileMapping, ._hFile1, NULL, PAGE_READONLY, 0, 0, NULL
		;
		; Save the file mapping handle for unmapping later
		;
		mov	._hFile1M, eax
		;
		; Map a view of the mapped file via MapViewOfFile, which takes five parameters:
		;
		;   1. Handle of mapped file
		;   2. Access mode (should be same as the file mapping)
		;   3. High 32 bits of offset of where view is to begin
		;   4. Low 32 bits
		;   5. Number of bytes for the view (zero = all)
		;
		; This is where we get access to the file and a view is like looking through a video
		; camera at the world - if we move the camera around we can see different parts of
		; the world, and if we zoom in and zoom out we can choose just how much of the world
		; we see at any one time. In the same way, we can choose the amount of the file to
		; view and wherabouts in the file the view should begin. For this example, I've
		; specified to view the whole file at once as this is a very convenient thing to do
		;
		sc MapViewOfFile, eax, FILE_MAP_READ, 0, 0, 0
		;
		; Save the mapping view address for unmapping later. Note that the return value is
		; the actual starting address of the view. Since we're viewing the whole file, this
		; address corresponds to the start of the file
		;
		mov	._hFile1V, eax		; eax -> start of file
		mov	ebx, eax		; ebx -> start of file
		add	ebx, ._ddFile1Size	; ebx -> end of file
		xor	ecx, ecx
		while	eax, l, ebx
			switch ecx
				case 0
					if byte [eax], e, '#'
						inc	ecx
					endif
					break
				case 1
					if byte [eax], e, 'd', byte [eax], e, 'D'
						inc	ecx
					endif
					break
				case 2
					if byte [eax], e, 'e', byte [eax], e, 'E'
						inc	ecx
					endif
					break
				case 3
					if byte [eax], e, 'f', byte [eax], e, 'F'
						inc	ecx
					endif
					break
				case 4
					if byte [eax], e, 'i', byte [eax], e, 'I'
						inc	ecx
					endif
					break
				case 5
					if byte [eax], e, 'n', byte [eax], e, 'N'
						inc	ecx
					endif
					break
				case 6
					if byte [eax], e, 'e', byte [eax], e, 'E'
						inc	ecx
					endif
					break
				case 7
					while byte [eax], e, ' ', byte [eax], e, 9
						inc	eax	; bypass spaces/tabs
					wend
					mov	edi, ._szLine	; edi -> buffer for this line
					mov	._ddLineSize, 0	; buffer is zero bytes at first
					;
					; note the use of whileand - we only want to loop if
					; the character isn't a space _and_ it isn't a tab
					;
					whileand byte [eax], ne, ' ', byte [eax], ne, 9
						mov	dl, [eax]
						inc	eax		; copy bytes from source to the buffer until
						mov	[edi], dl	; we hit a space or tab
						inc	edi
						inc	._ddLineSize	; increase buffer size
					wend
					mov	[edi], dword ' equ'	; add in ' equ '
					mov	[edi + 4], byte ' '
					add	edi, 5
					add	._ddLineSize, 5
					while byte [eax], e, ' ', byte [eax], e, 9
						inc	eax	; bypass spaces/tabs
					wend
					;
					; again note the use of whileand
					;
					whileand byte [eax], ne, ' ', byte [eax], ne, 9, byte [eax], ne, $0a, byte [eax], ne, $0d
						mov	dl, [eax]
						inc	eax		; copy bytes from source to the buffer until
						mov	[edi], dl	; we hit a space, tab or end of the line
						inc	edi
						inc	._ddLineSize	; increase buffer size
					wend
					mov	[edi], word $0a0d	; add a newline combination
					add	._ddLineSize, 2
					;
					; write this line to the output file, not forgetting to
					; save any registers we're using that will get trashed
					; during the API call
					;
					push	eax
					push	ecx
					sc WriteFile, ._hFile2, ._szLine, ._ddLineSize, .._ddLineSize, NULL
					pop	ecx
					pop	eax
					xor	ecx, ecx
					break
			switchend
			inc	eax		; onto the next character
		wend
		;
		; Before exiting, we need to close the file mapping and the files, but before
		; closing a file mapping, _all_ views need to be unmapped
		;
		sc UnmapViewOfFile, ._hFile1V
		sc CloseHandle, ._hFile1M
		sc CloseHandle, ._hFile2
		sc CloseHandle, ._hFile1
		sc ExitProcess, 0
	else
		TEXTlocal _szErrorNoFile, 'No input, or more than one, file specified.'
		TEXTlocal _szErrorNoFileTemp, 0
		mov	eax, .._szErrorNoFile
		sub	eax, ._szErrorNoFile
		sc WriteConsole, ._hStdOut, ._szErrorNoFile, eax, ._szErrorNoFileTemp, NULL
	endif
WinMain_end:
	sc ExitProcess, 0
endproc

[section .bss]

[section .data]
